@echo off
setlocal enabledelayedexpansion

title CLS Team - Service Host Process Optimization
echo CLS Team - Service Host Process Optimization
echo ============================================

:: Check if running as Administrator
net session >nul 2>&1
if %errorlevel% neq 0 (
    echo ERROR: This script must be run as Administrator!
    echo Right-click and select "Run as administrator"
    pause
    exit /b 1
)

echo Detecting system RAM...
set "totalRAM_GB="
set "detection_method="

:: Method 1: WMIC (most accurate)
for /f "tokens=2 delims==" %%I in ('wmic computersystem get TotalPhysicalMemory /value 2^>nul') do (
    if not defined totalRAM_GB (
        set /a totalRAM_GB=%%I / 1073741824
        set "detection_method=WMIC"
    )
)

:: Method 2: SystemInfo (fallback)
if not defined totalRAM_GB (
    for /f "tokens=2" %%I in ('systeminfo ^| findstr /C:"Total Physical Memory" 2^>nul') do (
        if not defined totalRAM_GB (
            for /f "tokens=1 delims=," %%J in ("%%I") do (
                set "ram_mb=%%J"
                set "ram_mb=!ram_mb:~0,-2!"
                set /a totalRAM_GB=!ram_mb! / 1024
                set "detection_method=SystemInfo"
            )
        )
    )
)

:: Method 3: PowerShell (final fallback)
if not defined totalRAM_GB (
    for /f "delims=" %%I in ('powershell -Command "[math]::Round((Get-CimInstance -ClassName Win32_ComputerSystem).TotalPhysicalMemory / 1GB)" 2^>nul') do (
        if not defined totalRAM_GB (
            set "totalRAM_GB=%%I"
            set "detection_method=PowerShell"
        )
    )
)

if not defined totalRAM_GB (
    echo ERROR: Cannot detect system RAM using any method.
    echo Please ensure you're running as Administrator.
    pause
    exit /b 1
)

echo [!detection_method!] Total RAM: !totalRAM_GB! GB

set "regPath=HKLM\SYSTEM\CurrentControlSet\Control"
set "valueName=SvcHostSplitThresholdInKB"
set "currentValue="
set "optimization_applied=0"

:: Check current registry value
for /f "tokens=3" %%I in ('reg query "%regPath%" /v "%valueName%" 2^>nul ^| findstr "%valueName%"') do (
    set "currentValue=%%I"
)

if defined currentValue (
    echo Current registry value: !valueName! = !currentValue!
) else (
    echo Current registry value: !valueName! is not set
)

:: Determine optimal threshold based on RAM
if !totalRAM_GB! geq 31 (
    set "threshold=33554432"
    set "ram_profile=32GB"
    set "optimization_applied=1"
) else if !totalRAM_GB! geq 15 (
    set "threshold=16777216" 
    set "ram_profile=16GB"
    set "optimization_applied=1"
) else if !totalRAM_GB! geq 7 (
    set "threshold=8388608"
    set "ram_profile=8GB"
    set "optimization_applied=1"
) else if !totalRAM_GB! geq 3 (
    set "threshold=4194304"
    set "ram_profile=4GB"
    set "optimization_applied=1"
) else (
    echo No optimization available for !totalRAM_GB! GB RAM
    echo Supported: 4GB, 8GB, 16GB, 32GB systems
    set "optimization_applied=0"
)

if !optimization_applied! equ 1 (
    echo.
    echo Configuring for !ram_profile! system...
    echo Setting !valueName! to !threshold!
    
    reg add "%regPath%" /v "%valueName%" /t REG_DWORD /d !threshold! /f >nul
    
    if !errorlevel! equ 0 (
        echo.
        echo SUCCESS: Registry value updated!
        echo Value: !valueName! = !threshold!
        echo RAM Profile: !ram_profile!
        echo.
        echo IMPORTANT: Reboot required for changes to take effect.
        echo.
        echo This optimization will reduce the number of svchost.exe processes
        echo by allowing more services to run in each shared process.
    ) else (
        echo.
        echo ERROR: Failed to update registry value.
        echo Ensure you have administrator privileges.
    )
)

echo.
echo Summary:
echo - RAM Detected: !totalRAM_GB! GB via !detection_method!
if defined ram_profile (
    echo - RAM Profile: !ram_profile!
)
if defined currentValue (
    echo - Previous Value: !currentValue!
)
if defined threshold (
    echo - New Value: !threshold!
)
echo - Optimization Applied: !optimization_applied!

echo.
pause